//
//  DataCenterTests.swift
//  Do It Tests
//
//  Created by Jim Dovey on 3/25/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import XCTest
import Combine
import Foundation
@testable import Do_It

extension TodoData {
    func unpack() -> (TodoItemList, TodoItem, UUID) {
        (lists[0], items[0], defaultListID)
    }
}

func randomColor() -> TodoItemList.Color {
    TodoItemList.Color.allPredefinedColors.randomElement()!
}

func randomIcon() -> String {
    listIconChoices.randomElement()!.randomElement()!
}

func createItems(_ count: Int, listID: UUID) -> [TodoItem] {
    (1...count).map {
        TodoItem(title: "Test Item \($0)", priority: .normal, listID: listID)
    }
}

func createLists(_ listCount: Int, withItems itemCount: Int) -> TodoData {
    var allItems: [TodoItem] = []
    let lists: [TodoItemList] = (1...listCount).map {
        let id = UUID()
        let items = createItems(itemCount, listID: id)
        allItems.append(contentsOf: items)
        return TodoItemList(id: id, name: "Test List \($0)", color: randomColor(),
                            icon: randomIcon(), items: items.map { $0.id })
    }
    return TodoData(lists: lists, items: allItems, defaultListID: lists[0].id)
}

let emptyTodoData = TodoData(lists: [], items: [], defaultListID: UUID(uuid: UUID_NULL))

class DataCenterTests: XCTestCase {
    private var subject = CurrentValueSubject<TodoData, Never>(emptyTodoData)
    private var subscribers: Set<AnyCancellable> = []
    
    private func createDataCenter(with data: TodoData) -> DataCenter {
        DataCenter(withTestItems: data,
                   scheduler: ImmediateScheduler.shared,
                   subject: subject,
                   subscribers: &subscribers)
    }

    override func setUpWithError() throws {
        subject.send(emptyTodoData)
    }

    override func tearDownWithError() throws {
        subscribers.removeAll()
    }

    func testRemoveItem() {
        let data = createLists(1, withItems: 1)
        let (list, item, _) = data.unpack()
        
        let dataCenter = DataCenter(withTestItems: data)
        XCTAssertEqual(dataCenter.todoLists.count, 1)
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 1)
        XCTAssertEqual(dataCenter.todoLists[0].items, [item.id])
        XCTAssertEqual(dataCenter.todoItems.count, 1)
        
        dataCenter.removeTodoItems(withIDs: [item.id])
        XCTAssertEqual(dataCenter.todoLists.count, 1)
        XCTAssertTrue(dataCenter.todoLists[0].items.isEmpty)
        XCTAssertTrue(dataCenter.items(in: list).isEmpty)
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
    }
    
    func testAddItem() {
        let data = createLists(1, withItems: 1)
        let (list, item, _) = data.unpack()
        let newItem = createItems(1, listID: list.id).first!
        
        let dataCenter = DataCenter(withTestItems: data)
        dataCenter.addTodoItem(newItem)
        
        XCTAssertEqual(dataCenter.todoLists[0].items, [item.id, newItem.id])
        XCTAssertEqual(dataCenter.todoItems.count, 2)
    }
    
    func testMoveItemsWithinList() {
        let data = createLists(1, withItems: 5)
        var itemIDs = data.items.map { $0.id }
        
        let dataCenter = DataCenter(withTestItems: data)
        XCTAssertEqual(dataCenter.todoLists.count, 1)
        XCTAssertEqual(dataCenter.todoLists[0].items, itemIDs)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        
        // move one item
        let move1 = IndexSet(integer: 4)
        dataCenter.moveTodoItems(fromOffsets: move1, to: 1,
                                 within: data.lists[0])
        // Items collection should not change
        XCTAssertEqual(dataCenter.todoItems, data.items)
        // Content of list's items array should change
        itemIDs.move(fromOffsets: move1, toOffset: 1)
        XCTAssertEqual(dataCenter.todoLists[0].items, itemIDs)
        
        // move two adjacent items
        let move2 = IndexSet(integersIn: 3...)
        dataCenter.moveTodoItems(fromOffsets: move2, to: 1,
                                 within: data.lists[0])
        XCTAssertEqual(dataCenter.todoItems, data.items)
        itemIDs.move(fromOffsets: move2, toOffset: 1)
        XCTAssertEqual(dataCenter.todoLists[0].items, itemIDs)
        
        // move two non-adjacent items
        let move3 = IndexSet([1,3])
        dataCenter.moveTodoItems(fromOffsets: move3, to: 2,
                                 within: data.lists[0])
        XCTAssertEqual(dataCenter.todoItems, data.items)
        itemIDs.move(fromOffsets: move3, toOffset: 2)
        XCTAssertEqual(dataCenter.todoLists[0].items, itemIDs)
    }
    
    func testMoveList() {
        let data = createLists(5, withItems: 1)
        var lists = data.lists
        
        let dataCenter = DataCenter(withTestItems: data)
        XCTAssertEqual(dataCenter.todoLists.count, 5)
        XCTAssertEqual(dataCenter.todoItems.count, 5)
        
        let move1 = IndexSet(integer: 4)
        dataCenter.moveLists(fromOffsets: move1, to: 2)
        lists.move(fromOffsets: move1, toOffset: 2)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists, lists)
        
        let move2 = IndexSet(integersIn: 3...)
        dataCenter.moveLists(fromOffsets: move2, to: 2)
        lists.move(fromOffsets: move2, toOffset: 2)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists, lists)
        
        let move3 = IndexSet([1, 3])
        dataCenter.moveLists(fromOffsets: move3, to: 2)
        lists.move(fromOffsets: move3, toOffset: 2)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists, lists)
    }
    
    func testMoveItemsAcrossLists() {
        let data = createLists(5, withItems: 1)
        let allItemIDs = data.items.map { $0.id }
        var dataCenter = DataCenter(withTestItems: data)
        
        // Test no overlaps
        let toMove1 = allItemIDs.dropFirst()
        dataCenter.moveTodoItems(withIDs: toMove1,
                                 toList: data.lists[0],
                                 at: 1)
        XCTAssertEqual(dataCenter.todoItems.map { $0.id }, data.items.map { $0.id })
        XCTAssertEqual(dataCenter.todoLists[0].items, allItemIDs)
        
        // Test all already in target list
        dataCenter.moveTodoItems(withIDs: toMove1, toList: data.lists[0],
                                 at: 1)
        XCTAssertEqual(dataCenter.todoLists[0].items, allItemIDs)
        
        // Test a mix of items.
        let data2 = createLists(5, withItems: 5)
        let idsToMove = data2.lists.indices.map { data2.lists[$0].items[$0] }
        var idsExpected = data2.lists[0].items
        idsExpected.insert(contentsOf: idsToMove, at: 3)
        idsExpected.removeFirst()
        
        dataCenter = DataCenter(withTestItems: data2)
        dataCenter.moveTodoItems(withIDs: idsToMove, toList: data2.lists[0],
                                 at: 3)
        XCTAssertEqual(dataCenter.todoLists[0].items, idsExpected)
        XCTAssertEqual(dataCenter.todoItems.map { $0.id },
                       data2.items.map { $0.id })
        XCTAssertNotEqual(dataCenter.todoItems.map { $0.listID },
                          data2.items.map { $0.listID })
    }
    
    func testUpdateLists() {
        let data = createLists(5, withItems: 1)
        let dataCenter = DataCenter(withTestItems: data)
        
        var lists = data.lists
        lists[0].name = "fred"
        lists[0].color = randomColor()
        lists[0].icon = randomIcon()
        
        dataCenter.updateList(lists[0])
        XCTAssertEqual(dataCenter.todoLists, lists)
        XCTAssertEqual(dataCenter.todoItems, data.items)
    }
    
    func testUpdateItems() {
        let data = createLists(5, withItems: 1)
        let dataCenter = DataCenter(withTestItems: data)
        
        // Change an item's details
        var items = data.items
        items[1].title = "fred"
        items[1].date = .distantFuture
        items[1].completed = Date()
        dataCenter.updateTodoItem(items[1])
        XCTAssertEqual(dataCenter.todoItems, items)
        
        // Now change that item's list ID
        items[1].listID = data.lists[0].id
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 1)
        dataCenter.updateTodoItem(items[1])
        XCTAssertEqual(dataCenter.todoItems.map { $0.id }, items.map { $0.id })
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 2)
    }
    
    func testRemoveList() {
        let data = createLists(2, withItems: 5)
        let dataCenter = DataCenter(withTestItems: data)
        
        let remainingItems = Array(data.items.dropFirst(5))
        dataCenter.removeLists(atOffsets: IndexSet(integer: 0))
        XCTAssertEqual(dataCenter.todoLists, Array(data.lists.dropFirst()))
        XCTAssertEqual(dataCenter.todoItems, remainingItems)
    }
    
    func testAutoDefaultList() {
        let data = createLists(1, withItems: 1)
        let dataCenter = DataCenter(withTestItems: data)
        
        XCTAssertEqual(dataCenter.defaultItemList, data.lists[0])
        
        dataCenter.removeLists(atOffsets: IndexSet(integer: 0))
        XCTAssertTrue(dataCenter.todoLists.isEmpty)
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
        
        let autoDefaultList = dataCenter.defaultItemList
        XCTAssertFalse(dataCenter.todoLists.isEmpty)
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
        XCTAssertEqual(dataCenter.todoLists, [autoDefaultList])
    }

    func testUndoRedoRemovals() throws {
        let data = createLists(1, withItems: 1)
        let dataCenter = DataCenter(withTestItems: data)
        let undoManager = try XCTUnwrap(dataCenter.undoManager)
        
        // remove the only item
        dataCenter.removeTodoItems(withIDs: [data.items[0].id])
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
        XCTAssertTrue(dataCenter.todoLists[0].items.isEmpty)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
        XCTAssertTrue(dataCenter.todoLists[0].items.isEmpty)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        // reset
        undoManager.undo()
        
        // remove the list, with its item
        dataCenter.removeLists(atOffsets: IndexSet(integer: 0))
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
        XCTAssertTrue(dataCenter.todoLists.isEmpty)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertTrue(dataCenter.todoItems.isEmpty)
        XCTAssertTrue(dataCenter.todoLists.isEmpty)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
    }
    
    func testUndoRedoAdditions() throws {
        let data = createLists(1, withItems: 1)
        let dataCenter = DataCenter(withTestItems: data)
        let undoManager = try XCTUnwrap(dataCenter.undoManager)
        
        let newItem = TodoItem(title: "New Item", priority: .normal,
                               listID: dataCenter.defaultListID)
        let newList = TodoItemList(name: "New List", color: randomColor(),
                                   icon: randomIcon())
        let newListItem = TodoItem(title: "New List Item", priority: .normal,
                                   listID: newList.id)
        
        // add item to existing list
        dataCenter.addTodoItem(newItem)
        XCTAssertEqual(dataCenter.todoItems.last, newItem)
        XCTAssertEqual(dataCenter.todoLists[0].items.last, newItem.id)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertEqual(dataCenter.todoItems.last, newItem)
        XCTAssertEqual(dataCenter.todoLists[0].items.last, newItem.id)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        let updatedLists = dataCenter.todoLists
        
        // add new list with no items
        dataCenter.addList(newList)
        XCTAssertEqual(dataCenter.todoLists.last, newList)
        XCTAssertEqual(dataCenter.todoItems.last, newItem)  // no extra items
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoLists, updatedLists)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertEqual(dataCenter.todoLists.last, newList)
        XCTAssertEqual(dataCenter.todoItems.last, newItem)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        // add new item to new list
        dataCenter.addTodoItem(newListItem)
        XCTAssertEqual(dataCenter.todoLists.last?.id, newList.id)
        XCTAssertNotEqual(dataCenter.todoLists.last, newList)   // items have changed
        XCTAssertEqual(dataCenter.todoItems.last, newListItem)
        XCTAssertEqual(dataCenter.todoLists.last?.items.last, newListItem.id)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoLists.last, newList)
        XCTAssertEqual(dataCenter.todoItems.last, newItem)
        XCTAssertTrue(dataCenter.todoLists.last!.items.isEmpty)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertEqual(dataCenter.todoLists.last?.id, newList.id)
        XCTAssertNotEqual(dataCenter.todoLists.last, newList)   // items have changed
        XCTAssertEqual(dataCenter.todoItems.last, newListItem)
        XCTAssertEqual(dataCenter.todoLists.last?.items.last, newListItem.id)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
    }
    
    func testUndoRedoMoves() throws {
        let data = createLists(5, withItems: 5)
        let dataCenter = DataCenter(withTestItems: data)
        let undoManager = try XCTUnwrap(dataCenter.undoManager)
        
        // move lists around
        dataCenter.moveLists(fromOffsets: IndexSet([3,5]), to: 1)
        XCTAssertNotEqual(dataCenter.todoLists, data.lists)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        let movedLists = dataCenter.todoLists
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertEqual(dataCenter.todoLists, movedLists)
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        // reset state
        undoManager.undo()
        undoManager.removeAllActions()
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        
        // move items in one list
        dataCenter.moveTodoItems(fromOffsets: IndexSet([3,5]), to: 1,
                                 within: data.lists[0])
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertNotEqual(dataCenter.todoLists[0], data.lists[0])
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoLists[0], data.lists[0])
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertNotEqual(dataCenter.todoLists[0], data.lists[0])
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        // reset state
        undoManager.undo()
        undoManager.removeAllActions()
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists[0].items[0], data.items[0].id)
        XCTAssertEqual(dataCenter.todoLists[1].items[0], data.items[5].id)
        
        // move one item across lists
        dataCenter.moveTodoItems(withIDs: [data.items[5].id],
                                 toList: data.lists[0], at: 0)
        XCTAssertNotEqual(dataCenter.todoLists[0], data.lists[0])
        XCTAssertNotEqual(dataCenter.todoLists[1], data.lists[1])
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 6)
        XCTAssertEqual(dataCenter.todoLists[1].items.count, 4)
        XCTAssertEqual(dataCenter.todoLists[0].items[0], data.items[5].id)
        XCTAssertEqual(dataCenter.todoLists[1].items[0], data.items[6].id)
        XCTAssertNotEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoItems.map { $0.id }, data.items.map { $0.id })
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertNotEqual(dataCenter.todoLists[0], data.lists[0])
        XCTAssertNotEqual(dataCenter.todoLists[1], data.lists[1])
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 6)
        XCTAssertEqual(dataCenter.todoLists[1].items.count, 4)
        XCTAssertEqual(dataCenter.todoLists[0].items[0], data.items[5].id)
        XCTAssertEqual(dataCenter.todoLists[1].items[0], data.items[6].id)
        XCTAssertNotEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoItems.map { $0.id }, data.items.map { $0.id })
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        // reset
        undoManager.undo()
        undoManager.removeAllActions()
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        
        // move items from many lists
        let itemIDs = data.lists[1...].compactMap { $0.items.first }
        dataCenter.moveTodoItems(withIDs: itemIDs, toList: data.lists[0], at: 0)
        XCTAssertNotEqual(dataCenter.todoLists, data.lists)
        XCTAssertNotEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 9)
        for i in 1..<5 {
            XCTAssertEqual(dataCenter.todoLists[i].items.count, 4, "List idx: \(i)")
            XCTAssertEqual(dataCenter.todoLists[i].items[0], data.lists[i].items[1], "List idx: \(i)")
            XCTAssertEqual(dataCenter.todoLists[0].items[i-1], data.lists[i].items[0], "List idx: \(i)")
        }
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
        
        undoManager.undo()
        XCTAssertEqual(dataCenter.todoLists, data.lists)
        XCTAssertEqual(dataCenter.todoItems, data.items)
        XCTAssertFalse(undoManager.canUndo)
        XCTAssertTrue(undoManager.canRedo)
        
        undoManager.redo()
        XCTAssertNotEqual(dataCenter.todoLists, data.lists)
        XCTAssertNotEqual(dataCenter.todoItems, data.items)
        XCTAssertEqual(dataCenter.todoLists[0].items.count, 9)
        for i in 1..<5 {
            XCTAssertEqual(dataCenter.todoLists[i].items.count, 4, "List idx: \(i)")
            XCTAssertEqual(dataCenter.todoLists[i].items[0], data.lists[i].items[1], "List idx: \(i)")
            XCTAssertEqual(dataCenter.todoLists[0].items[i-1], data.lists[i].items[0], "List idx: \(i)")
        }
        XCTAssertTrue(undoManager.canUndo)
        XCTAssertFalse(undoManager.canRedo)
    }
}
